include <rack_herringbone.scad>;
include <MCAD/motors.scad>;
include <MCAD/nuts_and_bolts.scad>;
include <MCAD/involute_gears.scad>;
include <MCAD/regular_shapes.scad>;

tolerance = 0.15;

// gear parameters
circular_pitch=350;
pressureAngle=25;
centerAngle=25;//angle at center of teeth
backlash=.4;
motorShaftDiameter=5+2*tolerance; // motor shaft

// magnet
magnetWidth=25.42+tolerance*2;
magnetLength=76.20+.4+tolerance*2;
magnetHeight=19.02+tolerance*2;

// guide rods
rodRad=3+tolerance;
linearBearingOD=12+tolerance*2;
linearBearingHeight=19+tolerance*2;


union(){
    translate([-80,0,0]){
        translate([-10,0,0])//
        color("blue") magnetPlate();
        color("red") pillar(30);

        translate([0,0,-20])//
        translate([40,0,-15])
            motorSide();
        
        translate([0,0,30])//
        color("orange")
        translate([40,0,2])
        rotate([0,180,360/10*-.15])
            pinion(10);
        
        color("green")
        translate([40,0,20])
        for (y = [-1,1]){
            translate([0, y*(magnetLength+5+2*(rodRad+3))/2, 0])
            bearingClamp();
        }
    }
    translate([10,0,0])//
        color("Purple") endPlate(30);
}


// Magnet plate
module magnetPlate(spacing = 30){
    thickness = 12;
    thicknessMin = 7;
    length = magnetLength+5;
    
    $fn=50;
    union(){
        difference(){// rod holders
            for (y = [-1,1]){ 
                translate([-thickness/2, y*(length+2*(rodRad+3))/2, 0])
                rotate([0,-90,0])
                    cube(size = [2*(rodRad+3),2*(rodRad+3), thickness], center = true);
            };
            
            for (y = [-1,1]){ // rod insert---halfway through holder + 1mm so face doesn't line up
                translate([1, y*(length+2*(rodRad+3))/2, 0])
                rotate([0,-90,0])
                    cylinder(h=thickness/2+1, r=rodRad);
            };
        }

        difference(){  //magnet trough
            translate([0, -length/2, -(magnetWidth+5)/2])
            rotate([0,-90,0]) 
                cube([magnetWidth+5, magnetLength+5, thickness]);
            
            translate([-thicknessMin, -magnetLength/2, -(magnetWidth)/2])
            rotate([0,-90,0])
                cube([magnetWidth, magnetLength, magnetHeight]);
            
            for (y = [-spacing/2, spacing/2]){
                translate([-thicknessMin+3, y, 0]) // cap recessed by 3mm
                rotate([0,+90,0])
                    boltHole(size=3, length=thicknessMin, extraCapHeight=10);
            }
        }
    }
}

module pillar(spacing = 30){
    thickness = 10;
    width = 50;
    length = 80;
    cavitySideThickness = 10;
    cavityEndThickness = 10; // shouldn't be smaller than 10 or nut hole is too far in
    
    // Rack parameters
    innerRadius=0;//shaft radius, in mm
    borders=5;//how thick should the borders around the central "shaft" be, in mm
    
    translate([0, -width/2, -thickness/2])
        difference(){
            cube([length,width,thickness]);
            for (r_z = [0,180]){
                translate([length/2, width/2, thickness/2])
                mirror([0,0,1]) //flip nut holes to top for easier printing
                rotate([0,0,r_z])
                translate([-length/2, -width/2, -thickness/2]) // center piece before rotation
                    for (y = [width/2-spacing/2, width/2+spacing/2]){ // M3 nut/bolt holes
                        translate([length-5, y, -1])
                        rotate([0,0,-90])
                        linear_extrude(height=thickness/2+3.2+1)
                            nutHole(size=3, proj=2);
                        
                        $fn=50;
                        translate([length-5, y, thickness/2])
                        rotate([0,90,0])
                            linear_extrude(height=6) boltHole(size=3, proj=1);
                    }
            }
        
            translate([cavityEndThickness, cavitySideThickness, -1])
                cube([length-cavityEndThickness*2, width-cavitySideThickness*2, thickness+2]);
    }
    
    translate([length/2+2, -width/2+7.3, -borders+2])
    rotate([0,0,180])
        rack(innerRadius, borders, 180/circular_pitch*20, 10, pressureAngle*pi/180, centerAngle);

}

module endPlate(spacing = 30){
    thickness = 10;
    length = magnetLength+5;
    $fn=50;
    
    union(){
        translate([0, -length/2, -thickness/2])
        difference(){
                cube([thickness, length, thickness]);
            for (y = [length/2-spacing/2, length/2+spacing/2]){
                translate([thickness-6, y, thickness/2]) // cap recessed by 6mm
                rotate([0,-90,0])
                    boltHole(size=3, length=thickness, extraCapHeight=10);
            }
        }
        
        difference(){// rod holders
            for (y = [-1,1]){ 
                translate([thickness/2, y*(magnetLength+5+2*(rodRad+3))/2, 0])
                rotate([0,90,0])
                    cube(size = [2*(rodRad+3),2*(rodRad+3), thickness], center = true);
            };
            
            for (y = [-1,1]){ // rod insert---halfway through holder + 1mm so face doesn't line up
                translate([-1, y*(magnetLength+5+2*(rodRad+3))/2, 0])
                rotate([0,90,0])
                    cylinder(h=thickness/2+1, r=rodRad);
            };
        }
    }
}

module motorSide(){
    $fn=50;
    thickness = 4;
    length = 50;
    width = (magnetLength+5)+(2*(rodRad+3))+(linearBearingOD+3);
    
    difference(){
        union(){
            difference(){
                union(){
                    difference(){ // main platform
                        translate([0,0,thickness/2])
                        cube([length, width, thickness], center = true);
                        
                        linear_extrude(height = thickness+1)
                        stepper_motor_mount(17, tolerance=tolerance);
                    }
                    
                    for (y = [-1, 1]){ // bearing blocks
                        translate([0, y*(magnetLength+5+2*(rodRad+3))/2, 15/2])
                            cube([linearBearingHeight+5, linearBearingOD+20, 15], center = true);
                    }
                }
            
                for (y = [-1, 1]){ // bolt holes
                        translate([-linearBearingHeight/2, y*(magnetLength+5+2*(rodRad+3))/2, 15])
                        union(){
                            rotate([0,90,0])
                                cylinder(h = linearBearingHeight, r = linearBearingOD/2); //bearing
                            translate([-3.5,0,0])
                            rotate([0,90,0])
                                cylinder(h = linearBearingHeight+7, r = rodRad+1); //shaft clearance
                            for (i = [-1, 1]){
                                translate([linearBearingHeight/2, i*(linearBearingOD/2+4), -16]) // bolts
                                    linear_extrude(height=15+2) boltHole(size = 3, proj = 1);
                                translate([linearBearingHeight/2, i*(linearBearingOD/2+4), -(15+12)+10]) // nuts
                                    linear_extrude(height=10+2) nutHole(size = 3, proj = 1);
                                
                            }
                        }
                }
            }
            
            rotate([0,0,180])
            translate([length/2-2, (magnetLength+5+2*(rodRad+3))/2, 0]){
                    translate([0, 0, (thickness)/2])
                        difference(){
                            cube([10.6, 19.8, thickness], center = true);
                            cube([11, 3, thickness+1], center = true);
                        }
            }
        }
        
        rotate([0,0,180])
        translate([length/2-2, (magnetLength+5+2*(rodRad+3))/2, thickness])
            microswitch(mockup = true, holeDepth = thickness+5, nutSpacing = 1.5);
    }
}

module bearingClamp(){
    $fn=50;
    
    difference(){
        translate([0, 0, (linearBearingOD/2+5)/2])
            cube([linearBearingHeight+5, linearBearingOD+20, linearBearingOD/2+5], center = true);
        
        translate([-linearBearingHeight/2, 0, 0])
        union(){
            rotate([0,90,0])
                cylinder(h = linearBearingHeight, r = linearBearingOD/2); //bearing
            
            translate([-3.5,0,0])
            rotate([0,90,0])
                cylinder(h = linearBearingHeight+7, r = rodRad+1); //shaft clearance
            
            for (i = [-1, 1]){
                translate([linearBearingHeight/2, i*(linearBearingOD/2+4), (linearBearingOD/2+5)-6]) // bolts
                rotate([0,180,0])
                    boltHole(size = 3, length = 15+2, proj = -1, extraCapHeight = 8);
                
            }
        }
    }
}

module pinion(teeth){
    //double helical gear
    $fn=50;
    twist=centerAngle;
    height=12;
    pressure_angle=pressureAngle;

    trap_width = 6;			//nut trap width across flats
    hardware_bore = 3.4;			//set screw clearence hole
    bore_diameter = motorShaftDiameter;

	//sets nut trap boss dimensions
	boss_length = 15/2 + 1;
	boss_width = trap_width + 3;
	trap_length = trap_width/2;
	
    mirror([0,0,1])
    gear (number_of_teeth=teeth,
        circular_pitch=circular_pitch,
        pressure_angle=pressure_angle,
        clearance = 0.2,
        gear_thickness = height/2,
        rim_thickness = height/2,
        rim_width = 10,
        hub_thickness = height/2,
        hub_diameter=15,
        bore_diameter=bore_diameter,
        circles=4,
        twist=twist,
        backlash=backlash);
    
	difference()
	{
		union()		//gear with boss for hardware on it
		{
		gear (number_of_teeth=teeth,
			circular_pitch=circular_pitch,
			pressure_angle=pressure_angle,
			clearance = 0.2,
			gear_thickness = height/2,
			rim_thickness = height/2,
			rim_width = 10,
			hub_thickness = height,
			hub_diameter=17,
			bore_diameter=bore_diameter,
			circles=4,
			twist=twist,
            backlash=backlash);
            
			translate([bore_diameter/2, -boss_width/2, height/2]) 	//center the cube
				cube([boss_length-bore_diameter/2, boss_width, height/2]); 	//boss the set screw lives in
		}

		union()	//set screw hardware removed parts
		{
			translate([bore_diameter/2+1.3, -trap_width/2, height-trap_width*1.2])
				cube([trap_length,trap_width,trap_width*1.3]);		//nut trap box

			translate([0, 0, height-0.6*trap_width])	//shift up
				rotate([0,90,0])							//lay flat
					cylinder(h=boss_length+1, r = hardware_bore/2);	//hardware hole
		}
	}
    
}
module microswitch(mockup = true, holeDepth = 0, nutSpacing = 0){
    $fn = 50;
    
    translate([-10.6/2, -19.8/2, 0]){
        if (mockup == true){
            %cube([10.6, 19.8, 6.4]);
        }
        
        for (y = [5.1,14.6]){
            translate([2.9, y, -holeDepth]){
                cylinder(h=6.4+holeDepth, r=2.5/2+tolerance); //M2.5 holes
                hexagon_prism(height = holeDepth-nutSpacing, radius = 2.55+tolerance);
            }
        }
    }
}